#############################################################
#                                                           #
#  ESPRESSO script                                          #
#                                                           #
#  Coarse-grained lipid model                               #
#  See details in Eur. Biophys. J. 35, 53 (2005)            #
#                                                           #
#  Alexander Lyubartsev    15.11.06                         #
#                                                           #
#  Lipid self-assembly simulation                           # 
#                                                           #
#############################################################

#
#   Begin with definition of a few subroutines
#

#  Dump XMOL file (use gOpenMol to view the trajectory)
#
proc confxmol { f_xmol } {
   global box_l n_part chem_typ
   puts $f_xmol [expr int($n_part)]
   set timefactor 63.
   set ttime [expr [setmd time] * $timefactor]  
   puts $f_xmol " after  $ttime fs,  BOX:  $box_l $box_l $box_l"
   for {set j 0} {$j < $n_part} {incr j} {
   puts $f_xmol " $chem_typ($j)  [part $j print pos] "
   }
}

#  Put molecules into the central box if they go out
#  Array moladr($n_mol) must be defined (= first atom of each molecule)
#  

proc ToPBC {} {
    global box_l n_mol moladr

# loop over molecules
    for {set i 0} {$i<$n_mol} {incr i} {
#  compute center-of-mass for each molecule
	set xcom 0
	set ycom 0
	set zcom 0
	set ii [expr $i + 1]
##	puts "$i $ii $moladr($i) $moladr($ii)"
	for {set j $moladr($i)} {$j < $moladr($ii)} { incr j } { 
	    set xcom [expr $xcom + [lindex [part $j print pos] 0 ] ]
            set ycom [expr $ycom + [lindex [part $j print pos] 1 ] ]
	    set zcom [expr $zcom + [lindex [part $j print pos] 2 ] ]
	}

	set xcom [ expr $xcom / ($moladr($ii)-$moladr($i))]
	set ycom [ expr $ycom / ($moladr($ii)-$moladr($i))]
	set zcom [ expr $zcom / ($moladr($ii)-$moladr($i))]
# find eventual shift
        if { $xcom < 0. } then {
           set xshift $box_l
        } elseif { $xcom > $box_l} {
	   set xshift [expr -1*$box_l]
	} else {
	   set xshift 0.
	}
        if { $ycom < 0. } then {
           set yshift $box_l
        } elseif { $ycom > $box_l } {
	   set yshift [expr -1*$box_l]
	} else {
	   set yshift 0.
	}
        if { $zcom < 0. } then {
           set zshift $box_l
        } elseif { $zcom > $box_l } {
	   set zshift [expr -1*$box_l]
	} else {
	   set zshift 0
	}
#  move the whole molecule so that its COM is inside the main cell
	for {set j $moladr($i)} {$j < $moladr($ii)} { incr j } { 
	  set xnew [expr [lindex [part $j print pos] 0] + $xshift]
	  set ynew [expr [lindex [part $j print pos] 1] + $yshift]
	  set znew [expr [lindex [part $j print pos] 2] + $zshift]
	  part $j pos $xnew $ynew $znew
	}
    }
}

#  --------------
#  The Main Unit
#  --------------


puts " "
puts "==================================================="
puts "=     Sample Script:  CGL.tcl                     ="
puts "=     Coarse Grained Lipid model                  ="
puts "==================================================="
puts " "

puts "Program Information: \n[code_info]\n"

#############################################################
#  Parameters                                               #
#############################################################

### System identification: 
set name  "CG_lipids"

###  Files

set rest_file "$name.dmp"
set traj_file "$name.1.xmol"
set obs_file "$name.1.obs"

###  Restart   read / write 
# 
set read_conf "no"
set rst_freq 100

###  observables
set obs_file [open $obs_file "w"]

#  names of .xmol and connectivity files for gOpenMol
set traj_file [open $traj_file "w"]
#  "connection file" is used by gOpenMol to draw a nicer picture (with bonds)
#  (run "source con.tcl" in the command line of gOpenMol) 
set conn_file [open con.tcl "w"]

#  For CPK presentation of atoms in gOpenMol:
#puts $conn_file "atom cpk * * * "
#puts $conn_file "atom scale cpk 2 * * * "
 
# System parameters
#
#############################################################
#
#  Units:   length = A
#           k = 1
#           m = 1 i.u.  
#           charge = e    
#           T = 1   (meaning 300 K)
#           E in kT units, that is E=1 is 2.5 kJ/M at 300k
#  unit time is computed as [t]^2 = [m]*[l]^2/[E]
#  if as above [l]=1A, [E]=1 kT, and we choose [m]=1 a.u.
#  then [t]=63 fs, (timestep 1 correspond to 63 fs)
#
###############################################################

#  Number of lipids
set N_L     400

#  Definition of one lipid
#  Num of CG sites
set n_sites 10
#  different sites
set n_part_types 4
##        at      x        y        z          M     q    typ
set A(1) {N     -3.748   -7.181   17.120      73.    1.     1}
set A(2) {P     -0.155   -7.093   15.977      122.  -1.     2}
set A(3) {O     -4.778   -5.353   12.085       72.   0.     3}
set A(4) {C     -8.314   -2.847    9.608       56.   0.     4}
set A(5) {C    -12.257   -0.932    7.098       56.   0.     4}
set A(6) {C    -14.702    0.863    3.142       57.   0.     4}
set A(7) {O     -3.711   -8.588   11.147       71.   0.     3}
set A(8) {C     -5.475   -6.660    7.570       56.   0.     4}
set A(9) {C     -8.849   -4.988    4.347       56.   0.     4}
set A(10) {C   -11.384   -3.096    0.539       57.   0.     4}

#  Box size
set box_l   200.

#  Bjerrum length
set bjerrum       7.1

# Integration parameters
#############################################################

# 
setmd temp      1.
setmd skin      1.
set shield      1.
# timestep and gamma may be different for warm-up and production
set warm_gamma     0.1
set warm_time_step 0.05
set md_gamma       0.02
set md_time_step   0.2

# warmup integration (with capped LJ potential) until particles are well apart
set warm_steps  200
set warm_loops  10
set warm_cap1   10
set warm_incr   1

# integration (with full tabulated potential) for $int_time
set int_steps  500
set int_times  2000

### others
set PI 3.14159256

#############################################################
#  Setup System                                             #
#############################################################

#  setting the box
setmd box_l $box_l $box_l $box_l

#  Defining interaction potentials
#
#  intramolecular bond potentials
inter 1 tabulated bond "NPint.pff"
inter 2 tabulated bond "POint.pff"
inter 3 tabulated bond "OCint.pff"
inter 4 tabulated bond "CCint.pff"
#  intermolecular potentials will be defined below

#  define starting positions of lipids

set i 0
set n_mol 0
set moladr($n_mol) $i

for { set ip 0 } { $ip < $N_L } { incr ip } {
# setup COMs
  set pocx [expr $box_l*([t_random]-0.5)]
  set pocy [expr $box_l*([t_random]-0.5)]
  set pocz [expr $box_l*([t_random]-0.5)]
# random rotation
# getting quaternion parameters
  set R1 [expr $PI*[t_random]*0.5]
  set R2 [expr $PI*[t_random]]
  set R3 [expr $PI*[t_random]]
  set Q1 [expr sin($R1) * sin([expr $R2-$R3])]
  set Q2 [expr sin($R1) * cos([expr $R2-$R3])]
  set Q3 [expr cos($R1) * sin([expr $R2+$R3])]
  set Q4 [expr cos($R1) * cos([expr $R2+$R3])]
# loop over lipid sites
  set i0 $i
  for { set j 1 } { $j <= $n_sites } { incr j } {
## rotate quaternion
    set dx [lindex $A($j) 1]
    set dy [lindex $A($j) 2]
    set dz [lindex $A($j) 3]
    set rx [expr $dx * (- $Q1*$Q1 + $Q2*$Q2 - $Q3*$Q3 + $Q4*$Q4 ) - \
2 * $dy * ($Q3 * $Q4 + $Q1 * $Q2 ) + 2 * $dz * ($Q2 * $Q3 - $Q1 * $Q4 )]
    set ry [expr $dy*($Q1*$Q1-$Q2*$Q2-$Q3*$Q3+$Q4*$Q4)+\
2*$dx*($Q3*$Q4-$Q1*$Q2)-2*$dz*($Q2*$Q4+$Q1*$Q3)]
    set rz [expr $dz*(-$Q1*$Q1-$Q2*$Q2+$Q3*$Q3+$Q4*$Q4)+\
2*$dx*($Q2*$Q3+$Q1*$Q4)+2*$dy*($Q2*$Q4-$Q1*$Q3)]
    set xx [expr $pocx + $rx]
    set yy [expr $pocy + $ry]
    set zz [expr $pocz + $rz]
    set pm [lindex $A($j) 4]
    set pq [lindex $A($j) 5]
    set pt [lindex $A($j) 6]
    part $i pos $xx $yy $zz type $pt q  $pq mass $pm
    set chem_typ($i) [lindex $A($j) 0]
    incr i
  }
#  setting bonds and exclude them from non-bonded interactions 
  part $i0 bond  1  [expr $i0+1]
  part $i0 exclude [expr $i0+1]
  puts $conn_file "edit bond create * * [expr $i0+1]  * * [expr $i0+2]"
  part [expr $i0+1] bond  2  [expr $i0+2]
  part [expr $i0+1] exclude  [expr $i0+2]
  puts $conn_file "edit bond create * * [expr $i0+2]  * * [expr $i0+3]"
  part [expr $i0+2] bond  3  [expr $i0+3]
  part [expr $i0+2] exclude  [expr $i0+3]
  puts $conn_file "edit bond create * * [expr $i0+3]  * * [expr $i0+4]"
  part [expr $i0+3] bond  4  [expr $i0+4]
  part [expr $i0+3] exclude  [expr $i0+4]
  puts $conn_file "edit bond create * * [expr $i0+4]  * * [expr $i0+5]"
  part [expr $i0+4] bond  4  [expr $i0+5]
  part [expr $i0+4] exclude  [expr $i0+5]
  puts $conn_file "edit bond create * * [expr $i0+5]  * * [expr $i0+6]"
  part [expr $i0+1] bond  2  [expr $i0+6]
  part [expr $i0+1] exclude  [expr $i0+6]
  puts $conn_file "edit bond create * * [expr $i0+2]  * * [expr $i0+7]"
  part [expr $i0+6] bond  3  [expr $i0+7]
  part [expr $i0+6] exclude  [expr $i0+7]
  puts $conn_file "edit bond create * * [expr $i0+7]  * * [expr $i0+8]"
  part [expr $i0+7] bond  4  [expr $i0+8]
  part [expr $i0+7] exclude  [expr $i0+8]
  puts $conn_file "edit bond create * * [expr $i0+8]  * * [expr $i0+9]"
  part [expr $i0+8] bond  4  [expr $i0+9]
  part [expr $i0+8] exclude  [expr $i0+9]
  puts $conn_file "edit bond create * * [expr $i0+9]  * * [expr $i0+10]"
#  count molecules
  incr n_mol
  set moladr($n_mol) $i
}

set n_part $i

puts " $n_part allocated"

#  initialize electrostatics

# this can be tuned: 

puts " initializing P3M ..."
    flush stdout
    flush $conn_file

setmd time_step $md_time_step
setmd gamma $md_gamma
#
#  to define optimal parameters, run with tuning
#
#inter coulomb $bjerrum p3m tunev2 accuracy 0.001

#  tuned parameters (for box size 200):
set r_cut          28.
set mesh           16
set cao            6
set alpha          0.068
inter coulomb $bjerrum p3m $r_cut $mesh $cao $alpha 

puts " Electrostatics parameters: [inter coulomb]"
    flush stdout

# read restart coordinates
if { $read_conf == "yes" } then {
  puts "trying to read restart file $rest_file"
  set restfile [open $rest_file "r"]
  blockfile $restfile read auto
  blockfile $restfile read particles
  close $restfile
  puts "Old configuration restarted successfully. Time [setmd time]"
  puts "\n "
  set warm_loops 0

} else {

# Ready for takeoff

  setmd time 0
  ToPBC
  confxmol $traj_file 
  puts " Start configuration written"
  puts " Ready to start"
 
##  warming up with cuped LJ potentials 
#  Set LJ cup

setmd time_step $warm_time_step
setmd gamma $warm_gamma

set lj_eps     1.
set lj_sig     5.
set lj_cut     [expr 1.12246*$lj_sig]
set lj_shift   [expr 0.25*$lj_eps]

set cap $warm_cap1
inter 1 1 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 1 2 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 1 3 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 1 4 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 2 2 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 2 3 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 2 4 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 3 3 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 3 4 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0
inter 4 4 lennard-jones $lj_eps $lj_sig $lj_cut $lj_shift 0

inter ljforcecap $cap

puts "Full list of warm-up interactions: \n [inter]"
puts " "


# Warmup Integration Loop
set i 0
while { $i < $warm_loops } {
    integrate $warm_steps
    set act_min_dist [analyze mindist]
    set act_energy [analyze energy]
    set bond_ener [expr [analyze energy bonded 1]+[analyze energy bonded 2]+\
		       [analyze energy bonded 3]+[analyze energy bonded 4]]
    puts "time=[setmd time] LJcap=$cap min dist = $act_min_dist \
     Ebond= $bond_ener  E= [lindex $act_energy 0 1] \r"
    flush stdout

#   write observables
    puts $obs_file "{ time [setmd time] } $act_energy"

#   Increase LJ cap
    set cap [expr $cap+$warm_incr]
    inter ljforcecap $cap
    incr i

#  put configuration in XMOL trajectory
    ToPBC
    confxmol $traj_file 

}

#  Preparing to normal MD

puts "\nChanging to tabulated CG potentials"
puts "\nStart integration: run $int_times times $int_steps steps"

# Change to intermolecular tabulated potential

inter ljforcecap 0
#  remove warm-up LJ potential
#
inter 1 1 lennard-jones  0 0 0 0 0
inter 1 2 lennard-jones  0 0 0 0 0
inter 1 3 lennard-jones  0 0 0 0 0
inter 1 4 lennard-jones  0 0 0 0 0
inter 2 2 lennard-jones  0 0 0 0 0
inter 2 3 lennard-jones  0 0 0 0 0
inter 2 4 lennard-jones  0 0 0 0 0
inter 3 3 lennard-jones  0 0 0 0 0
inter 3 4 lennard-jones  0 0 0 0 0
inter 4 4 lennard-jones  0 0 0 0 0
#
#  close the start-up part
}

#   Set tabulated intramolecular potentials
inter 1 1 tabulated "NN.pff"
inter 1 2 tabulated "NP.pff"
inter 1 3 tabulated "NO.pff"
inter 1 4 tabulated "NC.pff"
inter 2 2 tabulated "PP.pff"
inter 2 3 tabulated "PO.pff"
inter 2 4 tabulated "PC.pff"
inter 3 3 tabulated "OO.pff"
inter 3 4 tabulated "OC.pff"
inter 4 4 tabulated "CC.pff"

set act_energy [analyze energy]
    set bond_ener [expr [analyze energy bonded 1]+[analyze energy bonded 2]+\
		       [analyze energy bonded 3]+[analyze energy bonded 4]]
    set kin_en [lindex $act_energy 1 1] 
    set el_en [analyze energy coulomb] 
puts "Tab: Etot= [lindex $act_energy 0 1] Eb= $bond_ener Eel= $el_en Ek= $kin_en"

## puts "Full list of interactions: \n [inter]"

setmd time_step $md_time_step
setmd gamma $md_gamma

puts "New time step $md_time_step gamma $md_gamma"

set i 0

while { $i < $int_times } {
    integrate $int_steps
    set act_energy [analyze energy]
    set bond_ener [expr [analyze energy bonded 1]+[analyze energy bonded 2]+\
		       [analyze energy bonded 3]+[analyze energy bonded 4]]
    set tempr [expr [lindex $act_energy 1 1]/(1.5*[setmd n_part])] 
    puts "time=[setmd time] Etot= [lindex $act_energy 0 1] Ebond= $bond_ener temp= $tempr"
#   write observables
    puts $obs_file "{ time [setmd time] } $act_energy "
    flush $obs_file
    incr i
#  put configuration in XMOL trajectory
    ToPBC
    confxmol $traj_file
#  writing restart
    if {[expr fmod($i,$rst_freq)] == 0 } then {
      set restfile [open $rest_file "w"]
      blockfile $restfile write variable time
      blockfile $restfile write particles
      close $restfile
      puts "restart point written"
    }
}

# the end state
polyBlockWrite "$name.0000" all

exit

